unit XIconListBox;

{
  TXIconBox ver. 1.0 beta 1
  =========================
  IconBox descsendant for independent loading (ICO, DLL, EXE, ICL, NIL) and
  displaying of icons.

  Freeware.

  Copyright  1998 Roman Stedronsky, xster05@st.vse.cz.

  All rights reserved. You may use this software in an application
  without fee or royalty, provided this copyright notice remains intact.

}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Menus;

type
  TXIconListBox = class(TListBox)
  private
    { Private declarations }
//    FItems: TStrings;
    FAutoDisable: boolean;
    FNumberOfIcons: integer;
    FItemWidth: integer;
    FXIcons: integer;
    FYIcons: integer;
    FOnChange: TNotifyEvent; // Borland forgot this one in the parent, no idea why
    FOnFileChange: TNotifyEvent;

    { Routines that should only be used internally by component }
//    procedure FreeIcons;
    procedure ResetSize;
  protected
    { Routines for setting property values and updating affected items }
    procedure SetAutoDisable(Value: boolean);
    procedure SetXIcons(Value: integer);
    procedure SetYIcons(Value: integer);

    { Icon service routines }
    function GetIcon(const Index: integer): TIcon;

    { Owner drawing routines }
    procedure MeasureItem(Index: Integer; var Height: Integer); override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function SetIcon(FileName: string; const Index: integer; const Where: integer): boolean;
    function DeleteIcon(const Index: integer): boolean;
  published
    { Name of icon file to display }
    { If true, the combobox will be disabled when FileName does not exist }
    property AutoDisable: boolean read FAutoDisable write SetAutoDisable default true;
    { The number of icons in the file.  -1 if FileName is not valid.  }
    property NumberOfIcons: integer read FNumberOfIcons default -1;
    { Number of icons that are to be displayed in the listbox.  The width is modified  }
    { automatically when you change this property.                                     }
    property XIcons: integer read FXIcons write SetXIcons default 4;
    { Number of icons that are to be displayed in the listbox.  The height is modified }
    { automatically when you change this property.                                     }
    property YIcons: integer read FYIcons write SetYIcons default 1;
    { Useful if you have statics the reflect the number of icons, etc. }
    property OnFileChange: TNotifyEvent read FOnFileChange write FOnFileChange;

    { Protected properties in parent that we will make available to everyone }
    property Align;
    property Color;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property Enabled;
    property ItemIndex;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
  end;

procedure Register;

implementation

uses ShellAPI;

{ TXIconListBox Component }

constructor TXIconListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  { Set default values }
  Style := lbOwnerDrawFixed;
  ItemHeight := GetSystemMetrics(SM_CYICON) + 6;
  FItemWidth := GetSystemMetrics(SM_CXICON) + 6;
  Font.Name := 'Arial';
  Font.Height := ItemHeight;
  FAutoDisable := true;
  FNumberOfIcons := -1;
  FYIcons := 1;
  { By setting XIcons instead of FXIcons, the windows will get sized }
  XIcons := 4;
end;

destructor TXIconListBox.Destroy;
begin
//  FreeIcons;
  inherited Destroy;
end;

{ Free the icon resources we created. }
{
procedure TXIconListBox.FreeIcons;
var
  Index: integer;
begin
  for Index := 0 to Items.Count - 1 do
  begin
    TIcon(Items.Objects[Index]).Free; 						// Don't use GetIcon here!
    Items.Objects[Index] := nil;
  end;
  FNumberOfIcons := -1;
end;
}
{ Reset the size of the listbox to reflect changes in orientation and IconsDisplayed }

procedure TXIconListBox.ResetSize;
begin
  Height := ItemHeight * YIcons + GetSystemMetrics(SM_CYHSCROLL) + 1;
  Width := FItemWidth * XIcons + 2;
  Columns := XIcons;
(*  if Orientation = lbVertical then begin
    { Set height to hold the desired number of icons }
    Height := ItemHeight * IconsDisplayed + 2;
    { Set width to an icon plus a scrollbar }
    Width := FItemWidth + GetSystemMetrics(SM_CXVSCROLL) + 10;
    { Make sure we don't have any columns. }
    Columns := 0;
  end else begin
    { Set height to an icon plus a scrollbar }
    Height := ItemHeight + GetSystemMetrics(SM_CYHSCROLL) + 1;
    { Set width to hold the desired number of icons }
    Width := FItemWidth * IconsDisplayed + 2;
    { Set number of columns in the listbox to the desired number of icons }
    Columns := IconsDisplayed;
  end;*)
end;

{ Update the AutoDisable property }

procedure TXIconListBox.SetAutoDisable(Value: boolean);
begin
  { If it's the same, we don't need to do anything }
  if Value = FAutoDisable then exit;
  FAutoDisable := Value;
  { Update the enabled state of control based on new AutoDisable setting }
end;

{ Set the number of icons to be displayed in the listbox }

procedure TXIconListBox.SetXIcons(Value: integer);
begin
  { If number hasn't changed then don't do anything }
  if (Value = FXIcons) or (Value < 1) then exit;
  FXIcons := Value;
  { Call ResetSize to update the width or height, depending on the orientation }
  ResetSize;
end;

procedure TXIconListBox.SetYIcons(Value: integer);
begin
  { If number hasn't changed then don't do anything }
  if (Value = FYIcons) or (Value < 1) then exit;
  FYIcons := Value;
  { Call ResetSize to update the width or height, depending on the orientation }
  ResetSize;
end;

{ Used to extract icons from files and assign them to a TIcon object }

function TXIconListBox.SetIcon(FileName: string; const Index: integer; const Where: integer): Boolean;
var
  Icon		: TIcon;
begin
  Result := true;
  if (Where > (Items.Count - 1)) or (Items.Count = 0) then
  begin
    Icon := TIcon.Create;
    Icon.Handle := ExtractIcon(hInstance, PChar(FileName), Index);
    if Icon.Handle in [0, 1] then
    begin
      Result := false;
      Icon.Handle := LoadIcon(HInstance, IDI_APPLICATION);
    end;
    Items.AddObject(Format('%d', [Where]), Icon);
    if FNumberOfIcons = -1 then
      FNumberOfIcons := 1
    else
      Inc(FNumberOfIcons);
  end
  else
  begin
    DeleteIcon(Where);
    Icon := TIcon.Create;
    Icon.Handle := ExtractIcon(hInstance, PChar(FileName), Index);
    if Icon.Handle in [0, 1] then
    begin
      Result := false;
      Icon.Handle := LoadIcon(HInstance, IDI_APPLICATION);
    end;
    Items.InsertObject(Where, Format('%d', [Where]), Icon);
    if FNumberOfIcons = -1 then
      FNumberOfIcons := 1
    else
      Inc(FNumberOfIcons);
  end;
end;

{ Delete icon at Index position }

function TXIconListBox.DeleteIcon(const Index: integer): boolean;
begin
  Result := true;
  if Index <= Items.Count then		 // delete only existing icon
  begin
    TIcon(Items.Objects[Index]).Free;
    Items.Objects[Index] := nil;
    Items.Delete(Index);
    if FNumberOfIcons = 0 then
      FNumberOfIcons := -1
    else
      Dec(FNumberOfIcons);
  end
else
  Result := false;
end;

{ Returns the icon for a given combobox index }

function TXIconListBox.GetIcon(const Index: integer): TIcon;
begin
  { Return the requested icon }
  Result := TIcon(Items.Objects[Index]);
end;

{ Return the size of the item we are drawing }

procedure TXIconListBox.MeasureItem(Index: Integer; var Height: Integer);
begin
  { Ask Windows how tall icons are }
  Height := GetSystemMetrics(SM_CYICON);
end;

{ Draw the item requested in the given rectangle.  Because of the parent's default }
{ behavior, we needn't worry about the State.  That's very nice.                   }

procedure TXIconListBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
  Icon: TIcon;
begin
  { Use the controls canvas for drawing... }
  with Canvas do
  begin
    try
      { Fill in the rectangle.  The proper brush has already been set up for us,   }
      { so we needn't use State to set it ourselves.                               }
      FillRect(Rect);
      { Get the icon to be drawn }
      Icon := TIcon(Items.Objects[Index]); //GetIcon(Index);
      { If nothing has gone wrong, draw the icon.  Theoretically, it should never  }
      { be NIL, but why take the chance?                                           }
      if Icon <> nil then
 { Using the given rectangle, draw the icon on the control's canvas,        }
 { centering it within the rectangle.                                       }
        with Rect do
          Draw(Left + (Right - Left - Icon.Width) div 2,
            Top + (Bottom - Top - Icon.Width) div 2, Icon);
    except
      { If anything went wrong, we fall down to here.  You may want to add some    }
      { sort of user notification.  No clean up is necessary since we did not      }
      { create anything.  We'll just ignore the problem and hope it goes away. :)  }
      {!};
    end;
  end;
end;

{ Add the components to the Delphi Component Palette.  You will want to modify     }
{ this so that it appears on the page of your choice.                              }

procedure Register;
begin
  RegisterComponents('Extra', [TXIconListBox]);
end;

end.

